/************************************************************************
 * NAME:	fs-hdos.c
 *
 * DESCR:	Implements the Heathkit HDOS file system on top of the floppy struct.
 *
 *
 * NOTES:	
 ************************************************************************/
#include <sys/types.h>
#include <sys/stat.h>
#include <ctype.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <fcntl.h>
#include "getopt.h"

#include "standard.h"
#include "floppy.h"
#include "fs-hdos.h"


/************************************************************************
 * NAME:	fs_hdos_init() and fs_hdos_cleanup()
 *
 * DESCR:	Initializes/cleans-up the HDOS file system structures.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
fs_hdos_init(struct floppy *floppy, struct fs_hdos *fs)
{
    extern int floppy_read_sector(struct floppy *, int, int, int, char *);
    extern int floppy_write_sector(struct floppy *, int, int, int, char *);

    if (!hdos_init(&fs->hdosfs,floppy,floppy_read_sector,floppy_write_sector)) {
	return(FALSE);
    }

    return (hdos_settings(&fs->hdosfs,0,5));
}

int
fs_hdos_cleanup(struct fs_hdos *fs)
{
    return(hdos_cleanup(&fs->hdosfs));
}

/************************************************************************
 * NAME:	fs_hdos_report()
 *
 * DESCR:	Generates a report on the filesystem structure.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
fs_hdos_report(struct fs_hdos *fs, int verbosity)
{
    char	buffer[1000];

    buffer[0] = '\0';
    printf("%s",hdos_label_report(&fs->hdosfs,buffer,verbosity));

    hdos_dir_report(&fs->hdosfs,verbosity);

    return(TRUE);
}

/************************************************************************
 * NAME:	fs_hdos_add()
 *
 * DESCR:	Adds the referenced file to the HDOS image.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
fs_hdos_add(struct fs_hdos *fs, char *name, int unusedfd)
{
    struct hdos_file	*file;
    char		 buffer[BUFSIZ];
    int			 i;
    hdos_inode		 inode;
    int			 fd;
    struct stat		 statbuf;
    int			 needed_sectors;

    /* check free space first	*/

    fd = open(name,O_RDONLY);
    if (fd < 0) {
	return(FALSE);
    }

    fstat(fd,&statbuf);

    if (statbuf.st_size == 0) {
	needed_sectors = 1;
    } else {
	needed_sectors = (statbuf.st_size - 1) / HDOS_SECTSIZE(&fs->hdosfs) + 1;
    }

    if (needed_sectors <= hdos_grt_free_groups(&fs->hdosfs)*HDOS_GROUPSIZE(&fs->hdosfs) ) {

	inode = hdos_dir_find(&fs->hdosfs,name);

	if (inode == HDOS_INODE_NULL) {

	    file = hdos_file_new(&fs->hdosfs,name);

	    if (file != NULL) {
		while ( i = read(fd,buffer,BUFSIZ) ) {
		    if (!hdos_file_write(file,buffer,i)) {
			break;
		    }
		}

		hdos_file_close(file);

		if (hdos_dir_write(&fs->hdosfs)) {
		    if (hdos_grt_write(&fs->hdosfs)) {
			if (hdos_rgt_write(&fs->hdosfs)) {
			    close(fd);
			    return(TRUE);
			}
		    }
		}
	    }
	}
    }
    close(fd);
    return(FALSE);
}

/************************************************************************
 * NAME:	fs_hdos_del()
 *
 * DESCR:	Deletes the named file from the filesystem.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
fs_hdos_del(struct fs_hdos *fs, char *name)
{
    hdos_inode	inode;

    inode = hdos_dir_find(&fs->hdosfs,name);

    if (inode == HDOS_INODE_NULL) {
	return(FALSE);
    }

    hdos_dir_filedel(&fs->hdosfs,inode);

    if (!hdos_dir_write(&fs->hdosfs)) {
	return(FALSE);
    }
    if (!hdos_grt_write(&fs->hdosfs)) {
	return(FALSE);
    }
    if (!hdos_rgt_write(&fs->hdosfs)) {
	return(FALSE);
    }

    return(TRUE);
}

/************************************************************************
 * NAME:	fs_hdos_extract()
 *
 * DESCR:	Extracts the named file from the filesystem, sending it
 *		out
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
fs_hdos_extract(struct fs_hdos *fs, char *name, int fd)
{
    struct hdos_file	*file;
    char		 buffer[BUFSIZ];
    int			 output;
    int			 i;
    hdos_inode		 inode;

    inode = hdos_dir_find(&fs->hdosfs,name);

    if (inode == HDOS_INODE_NULL) {
	return(FALSE);
    }

    if ((output = open(name,O_CREAT|O_WRONLY|O_TRUNC,0777)) < 0) {
	return(FALSE);
    }

    file = hdos_file_open(&fs->hdosfs,name);

    if (file == NULL) {
	return(FALSE);
    }

    while ( i = hdos_file_read(file,buffer,BUFSIZ) ) {
	write(output,buffer,i);
    }

    hdos_file_close(file);
    close(output);

    return(TRUE);
}

/************************************************************************
 * NAME:	fs_hdos_description()
 *
 * DESCR:	Returns the description of this type of file system.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
char *
fs_hdos_description(void)
{
    return("H8/H89 HDOS");
}


/************************************************************************
 * NAME:	fs_hdos_compress()
 *
 * DESCR:
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:
 ************************************************************************/
int
fs_hdos_compress(struct fs_hdos *fs)
{
}
